﻿import argparse
import os
import pandas as pd
import matplotlib.pyplot as plt

def main():
    ap = argparse.ArgumentParser(description="Plot dependency accretion curve from Federal Register baseline CSV.")
    ap.add_argument("--input", required=True, help="Path to federal_registry_baseline_1994_2025.csv")
    ap.add_argument("--output", required=True, help="Output PNG path")
    args = ap.parse_args()

    if not os.path.exists(args.input):
        raise SystemExit(f"ERROR: Input CSV not found: {args.input}")

    df = pd.read_csv(args.input)

    required = {"date", "dependency_rate"}
    missing = required - set(df.columns)
    if missing:
        raise SystemExit(f"ERROR: Missing required columns: {missing}")

    df["date"] = pd.to_datetime(df["date"], errors="coerce")
    df = df.dropna(subset=["date"]).sort_values("date")

    # simple rolling mean to show long-run accretion (no scoring, just smoothing)
    df["dep_roll"] = df["dependency_rate"].rolling(window=90, min_periods=30).mean()

    plt.figure()
    plt.plot(df["date"], df["dependency_rate"], linewidth=0.7)
    plt.plot(df["date"], df["dep_roll"], linewidth=1.5)
    plt.title("Federal Register (1994–2025): Dependency Rate Over Time")
    plt.xlabel("Date")
    plt.ylabel("Dependency rate")
    plt.tight_layout()

    outdir = os.path.dirname(os.path.abspath(args.output))
    if outdir and not os.path.exists(outdir):
        os.makedirs(outdir, exist_ok=True)

    plt.savefig(args.output, dpi=200)
    print(f"OK: wrote {args.output}")

if __name__ == "__main__":
    main()
